# Copyright (c) 2015, R. Algar,  F. Ceroni, T. Ellis, G.-B. Stan
# All rights reserved.

# Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

# 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.

# 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.

# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


from scipy.optimize import fsolve, root
import numpy as np

###############################
# Ribosomal Competition Model #
###############################

class Cell(object):
    """
    Object respresenting a cell.
    
    free_ribosomes = integer number of ribosomes available in the cell 
        for synthetic circuits to use.
    
    circuits = list of synthetic gene circuits in cell. To simulate a cell
        with a burden monitor, this must be included in this list of circuits.
        This is a list of instances of the Circuit class.
    
    """
    def __init__(self, free_ribosomes=10000, circuits=[]):
        self.free_ribosomes = free_ribosomes
        self.circuits = circuits
        self.array = [0] #free ribosomes
        for circuit in circuits:
            self.array += [0]
    
    def conservation(self,p):
        return (sum(p)-self.free_ribosomes,)

    def equation(self, y):
        # Calculate length of vector
        vlen = len(self.initial_conditions())
        # Create initial equation vector
        eqn = np.zeros(vlen)
        # Set initial length
        L = 0
        # Iterate over circuits
        for circuit in self.circuits:
            M = circuit.total_transcripts
            a_p = circuit.alpha_plus
            a_m = circuit.alpha_minus
            betas = circuit.betas
            RBS_pos = L + 1
            circ_len = circuit.length
            fin_pos = RBS_pos + circ_len
            ## Free ribosome pool dynamics
            eqn[0] = self.conservation(y)[0]
            # RBS dynamics
            eqn[RBS_pos] = ( M*a_p*y[0]*(1-y[RBS_pos]/M)) - a_m*y[RBS_pos] - betas[0]*y[RBS_pos]*(1-y[RBS_pos+1]/M)
            # Final codon
            eqn[fin_pos] = betas[circ_len-1]*y[fin_pos-1]*(1-y[fin_pos]/M) - betas[circ_len]*y[fin_pos]
            # Iterate along transcript
            for i in range(circ_len-1):
                j = i+1
                eqn[RBS_pos+j] = betas[j-1]*y[RBS_pos+j-1]*(1-y[RBS_pos+j]/M) - betas[j]*y[RBS_pos+j]*(1-y[RBS_pos+j+1]/M)
            L = fin_pos
        return eqn

    def initial_conditions(self):
        '''
        Default initial conditions if none provided
        '''
        initial_conditions = [0]
        for circuit in self.circuits:
            initial_conditions += circuit.initial_conditions
        initial_conditions[0] = self.free_ribosomes - sum(initial_conditions)
        return initial_conditions
    
    def simulate(self, ic=[]):
        '''
        Simulates steady state equation of ribosomal distribution over all circuti transcripts using scipy.optimize.root
        '''
        # Check if initial conditions have been provided
        if len(ic) > 0:
            initial_conditions = ic
        else:
            initial_conditions = self.initial_conditions()
            print 'Setting initial conditions:', initial_conditions
        #solution, infodict, ier, mesg = fsolve(self.equation, initial_conditions, full_output=True)
        #print initial_conditions
        # Set method for finding zero-point
        method = 'lm'
        sol = root(self.equation, initial_conditions, method=method, jac=self.jacobian)
        # Print message to console to indicate whether it has been a success
        print sol['success'], sol['message']
        # Set solution as the 'x' of the solution dictionary
        solution = sol['x']
        #result = {'free_ribosomes': solution[0], 'circuits': [], 'success_code': ier}
        result = {'free_ribosomes': solution[0], 'circuits': [], 'success_code': sol['success']}
        shift_counter = 1
        for circuit in self.circuits:
            length = circuit.length+1
            result['circuits'] += [solution[shift_counter:shift_counter +
                                    length]]
            shift_counter += length
        return result, solution

    def jacobian(self, y):
        # Calculate dimensions of jacobian matrix
        dim = len(self.initial_conditions())
        # Create initial jacobian matrix
        jac = np.zeros((dim, dim))
        # Set initial length
        L = 0
        # Iterate over circuits
        for circuit in self.circuits:
            M = circuit.total_transcripts
            a_p = circuit.alpha_plus
            a_m = circuit.alpha_minus
            betas = circuit.betas
            RBS_pos = L + 1
            circ_len = circuit.length
            fin_pos = RBS_pos + circ_len
            # Free ribosome pool dynamics
            #jac[0,0] += - ( M*a_p*(1-y[RBS_pos]/M))
            #jac[0,RBS_pos] = a_p*y[0] + a_m
            #jac[0,fin_pos] = betas[circ_len]
            jac[0,:] = 1
            # RBS dynamics
            jac[RBS_pos,0] = M*a_p*(1-y[RBS_pos]/M)
            jac[RBS_pos,RBS_pos] = - a_p*y[0] - a_m - betas[0]*(1-y[RBS_pos+1]/M)
            jac[RBS_pos,RBS_pos+1] = betas[0]*y[RBS_pos]/M
            # Final codon
            jac[fin_pos,fin_pos-1] = betas[circ_len-1]*(1-y[fin_pos]/M)
            jac[fin_pos,fin_pos] = betas[circ_len-1]*y[fin_pos-1]/M - betas[circ_len]
            # Iterate along transcript
            for i in range(circ_len-1):
                j = i+1
                jac[RBS_pos+j,RBS_pos+j-1] = betas[j-1]*(1-y[RBS_pos+j]/M)
                jac[RBS_pos+j,RBS_pos+j] = - betas[j-1]*y[RBS_pos+j-1]/M - betas[j]*(1-y[RBS_pos+j+1]/M)
                jac[RBS_pos+j,RBS_pos+j+1] = betas[j]*y[RBS_pos+j]/M
            L = fin_pos
        return jac


class Circuit(object):
    """
    Object representing synthetic gene circuits that will be placed into
    a cell.

    total_transcripts = integer number of transcripts for this circuit
        in the cell. Is a function of both copy number and promoter
        strength.

    alpha_plus = rate at which free ribosomes bind to RBS.

    alpha_minus = rate at which ribosomes unbind from RBS.

    betas = list of rates for ribosomes moving along transcript. betas[0]
        represents the rate at which ribosome moves from RBS to initial
        elongation state. betas[i] represents rate at which ribosomes 
        moves from position i to position i+1 if unblocked (or into free 
        ribosome pool for i = length(betas)).

    RBS_strength = single number that replaces alpha_plus, alpha_minus
        and betas[0] if defined.

    """
    def __init__(self, total_transcripts, alpha_plus_scale=0.0001, 
                 alpha_minus_scale=200, length=50, betas=None,
                 RBS_strength=1):
        self.total_transcripts = total_transcripts
        self.input_betas = betas
        if betas:
            self.betas = betas
            self.length = len(betas)-1
        else:
            self.betas = [1 for i in range(length+1)]
            self.length = length
        if RBS_strength:
            self.betas[0] = RBS_strength
        self.ap_scale = alpha_plus_scale
        self.am_scale = alpha_minus_scale
        self.RBS_strength = RBS_strength
        self.alpha_plus = alpha_plus_scale*self.betas[0]
        self.alpha_minus = alpha_minus_scale/self.betas[0]
        self.initial_conditions = [0 for i in range(self.length+1)]

